#!/usr/bin/env python3
"""
make_kernel_from_eigs.py

Utility to construct the base kernel required by the simulation pipeline
from the eigenvalue spectrum stored in ``kernel_eigs.csv``.  The script
reads the ``rho`` column from the CSV, tiles it as necessary to match the
expected number of lattice links (2 * L^2) and saves the resulting
one‑dimensional array to ``kernel.npy``.  The lattice size L is
inferred from the repository's configuration file.

This file is copied verbatim from the available
``vol4-discrete-gauge-wilson-loop`` repository and must not be
modified.  It is placed under ``kernel_builder/`` in this project to
construct a realistic kernel for subsequent simulations.
"""

import os
import argparse
import yaml
import numpy as np
import pandas as pd


def determine_lattice_size(cfg: dict) -> int:
    """Extract the lattice size L from a configuration mapping.

    The configuration may define ``lattice_size`` at the top level or
    nested under a ``parameters`` section (as used in other
    repositories).
    """
    if 'lattice_size' in cfg:
        return int(cfg['lattice_size'])
    if 'parameters' in cfg and 'lattice_size' in cfg['parameters']:
        return int(cfg['parameters']['lattice_size'])
    raise KeyError('lattice_size not found in configuration')


def main():
    """Generate a real‑space kernel from an eigenvalue spectrum.

    The original implementation saved a one‑dimensional array of raw
    eigenvalues.  For the second‑pass volume sweep we require a
    full square kernel of dimension ``num_links × num_links``.  This
    routine reads eigenvalues from ``kernel_eigs.csv``, exponentiates
    them (as one would exponentiate a diagonal matrix), tiles the
    resulting spectrum to match the desired size, constructs a
    diagonal matrix and writes it to ``kernel.npy``.

    Parameters
    ----------
    --config : str, optional
        Path to a YAML file specifying the lattice size.  Defaults to
        ``config.yaml``.
    --input_csv : str, optional
        Input CSV containing an eigenvalue column labelled ``rho``.
    --output : str, optional
        Destination for the generated ``.npy`` file.
    """
    parser = argparse.ArgumentParser(description='Generate kernel.npy from kernel_eigs.csv')
    parser.add_argument('--config', default='config.yaml', help='Path to config file')
    parser.add_argument('--input_csv', default=os.path.join('data', 'kernel_eigs.csv'), help='Input CSV file')
    parser.add_argument('--output', default=os.path.join('data', 'kernel.npy'), help='Output .npy file')
    args = parser.parse_args()

    # Load configuration and determine lattice size L
    with open(args.config) as f:
        cfg = yaml.safe_load(f)
    L = determine_lattice_size(cfg)
    num_links = 2 * L * L

    # Read eigenvalues from CSV.  We look for a column named 'rho'
    # first; if it is absent we fall back to the last numeric column.
    df = pd.read_csv(args.input_csv)
    if 'rho' in df.columns:
        rho_vals = df['rho'].to_numpy(dtype=float)
    else:
        numeric_cols = df.select_dtypes(include=['float', 'int']).columns
        rho_vals = df[numeric_cols[-1]].to_numpy(dtype=float)

    # The eigenvalues represent logarithms of the kernel entries.  To
    # obtain the kernel we exponentiate the diagonal matrix.  For a
    # diagonal matrix this is equivalent to exponentiating each
    # eigenvalue individually.  We avoid using ``scipy.linalg.expm``
    # here because the matrix is diagonal and ``numpy.exp`` suffices.
    exp_vals = np.exp(rho_vals)

    # Tile the exponentiated eigenvalues to fill the diagonal of the
    # full kernel.  The kernel dimension must be ``num_links × num_links``.
    repeats = (num_links + len(exp_vals) - 1) // len(exp_vals)
    diag_entries = np.tile(exp_vals, repeats)[:num_links]

    # Construct a diagonal matrix.  Off‑diagonal elements are zero.
    kernel_matrix = np.zeros((num_links, num_links), dtype=float)
    np.fill_diagonal(kernel_matrix, diag_entries)

    # Save the kernel matrix to disk
    out_path = args.output
    # Ensure output directory exists
    os.makedirs(os.path.dirname(out_path), exist_ok=True)
    np.save(out_path, kernel_matrix)
    print(f'Saved kernel to {out_path} with shape {kernel_matrix.shape} and dtype {kernel_matrix.dtype}')


if __name__ == '__main__':
    main()